<?php
/**
 * نظام استيراد الصور الشخصية من Discord و Google
 * 
 * @package AT_Studio_Auth
 */

if (!defined('ABSPATH')) exit;

class AT_Studio_Avatar_Importer {
    
    /**
     * استيراد صورة الملف الشخصي وتحديثها للمستخدم
     */
    public static function import_avatar($user_id, $avatar_url, $provider = 'discord') {
        // التحقق من تفعيل خيار استيراد الصور
        $import_enabled = get_option('at_studio_import_avatar', true);
        if (!$import_enabled) {
            return false;
        }
        
        // الحصول على بيانات الصورة
        $response = wp_remote_get($avatar_url);
        
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }
        
        $image_data = wp_remote_retrieve_body($response);
        $file_name = 'avatar_' . $user_id . '_' . time() . '.jpg';
        
        // رفع الصورة إلى مكتبة الوسائط
        $upload = wp_upload_bits($file_name, null, $image_data);
        
        if ($upload['error']) {
            return false;
        }
        
        // إنشاء مرفق للصورة
        $attachment = array(
            'post_mime_type' => 'image/jpeg',
            'post_title'     => sanitize_file_name($file_name),
            'post_content'   => '',
            'post_status'    => 'inherit'
        );
        
        $attach_id = wp_insert_attachment($attachment, $upload['file']);
        
        if (is_wp_error($attach_id)) {
            return false;
        }
        
        // إنشاء بيانات المرفق
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        $attach_data = wp_generate_attachment_metadata($attach_id, $upload['file']);
        wp_update_attachment_metadata($attach_id, $attach_data);
        
        // تحديث صورة الملف الشخصي للمستخدم
        update_user_meta($user_id, 'at_studio_imported_avatar_id', $attach_id);
        update_user_meta($user_id, 'at_studio_avatar_provider', $provider);
        
        // تحديث صورة الملف الشخصي الرئيسية إذا كانت مطلوبة
        $set_as_profile = get_option('at_studio_set_imported_as_profile', true);
        if ($set_as_profile) {
            update_user_meta($user_id, 'wp_user_avatar', $attach_id);
        }
        
        return $attach_id;
    }
    
    /**
     * الحصول على صورة الملف الشخصي المستوردة
     */
    public static function get_imported_avatar($user_id, $size = 'thumbnail') {
        $avatar_id = get_user_meta($user_id, 'at_studio_imported_avatar_id', true);
        
        if ($avatar_id) {
            $avatar_url = wp_get_attachment_image_url($avatar_id, $size);
            return $avatar_url ?: false;
        }
        
        return false;
    }
    
    /**
     * حذف الصورة المستوردة
     */
    public static function delete_imported_avatar($user_id) {
        $avatar_id = get_user_meta($user_id, 'at_studio_imported_avatar_id', true);
        
        if ($avatar_id) {
            wp_delete_attachment($avatar_id, true);
            delete_user_meta($user_id, 'at_studio_imported_avatar_id');
            delete_user_meta($user_id, 'at_studio_avatar_provider');
            
            // إعادة تعيين صورة الملف الشخصي إذا كانت هي المستوردة
            $current_avatar = get_user_meta($user_id, 'wp_user_avatar', true);
            if ($current_avatar == $avatar_id) {
                delete_user_meta($user_id, 'wp_user_avatar');
            }
            
            return true;
        }
        
        return false;
    }
    
    /**
     * تحديث صورة الملف الشخصي من المصدر الأصلي
     */
    public static function refresh_avatar($user_id) {
        $provider = get_user_meta($user_id, 'at_studio_avatar_provider', true);
        $original_avatar = get_user_meta($user_id, 'at_studio_auth_avatar', true);
        
        if ($provider && $original_avatar) {
            // حذف الصورة القديمة أولاً
            self::delete_imported_avatar($user_id);
            
            // استيراد الصورة الجديدة
            return self::import_avatar($user_id, $original_avatar, $provider);
        }
        
        return false;
    }
    
    /**
     * معالجة استيراد الصور للمستخدمين الحاليين
     */
    public static function process_existing_users() {
        $users = get_users(array(
            'meta_key' => 'at_studio_auth_avatar',
            'meta_compare' => 'EXISTS'
        ));
        
        $results = array(
            'success' => 0,
            'failed' => 0,
            'total' => count($users)
        );
        
        foreach ($users as $user) {
            $avatar_url = get_user_meta($user->ID, 'at_studio_auth_avatar', true);
            $provider = get_user_meta($user->ID, 'at_studio_avatar_provider', true) ?: 'discord';
            
            if ($avatar_url && !get_user_meta($user->ID, 'at_studio_imported_avatar_id', true)) {
                $result = self::import_avatar($user->ID, $avatar_url, $provider);
                if ($result) {
                    $results['success']++;
                } else {
                    $results['failed']++;
                }
            }
        }
        
        return $results;
    }
}

// إضافة وظائف مساعدة
function at_studio_import_user_avatar($user_id, $avatar_url, $provider = 'discord') {
    return AT_Studio_Avatar_Importer::import_avatar($user_id, $avatar_url, $provider);
}

function at_studio_get_user_imported_avatar($user_id, $size = 'thumbnail') {
    return AT_Studio_Avatar_Importer::get_imported_avatar($user_id, $size);
}

function at_studio_delete_user_imported_avatar($user_id) {
    return AT_Studio_Avatar_Importer::delete_imported_avatar($user_id);
}

function at_studio_refresh_user_avatar($user_id) {
    return AT_Studio_Avatar_Importer::refresh_avatar($user_id);
}
