<?php
/**
 * نظام مصادقة Discord لـ WordPress
 * 
 * @package AT_Studio_Auth
 */

if (!defined('ABSPATH')) exit;

class AT_Studio_Discord_Auth {
    // بيانات تطبيق Discord
    private $client_id;
    private $client_secret;
    private $redirect_uri;

    public function __construct() {
        $this->client_id = get_option('at_studio_discord_client_id');
        $this->client_secret = get_option('at_studio_discord_client_secret');
        $this->redirect_uri = site_url('/?action=discord-callback');
    }

    /**
     * الحصول على رابط مصادقة Discord
     */
    public function get_auth_url() {
        $endpoint = 'https://discord.com/api/oauth2/authorize';
        $scopes = 'identify email';

        return $endpoint . '?' . http_build_query([
            'client_id' => $this->client_id,
            'redirect_uri' => $this->redirect_uri,
            'response_type' => 'code',
            'scope' => $scopes
        ]);
    }

    /**
     * معالجة رد المصادقة
     */
    public function handle_callback($code) {
        // التحقق من وجود بيانات الاعتماد
        if (empty($this->client_id) || empty($this->client_secret)) {
            wp_die(__('بيانات تطبيق Discord غير مكتملة. يرجى التحقق من الإعدادات.', 'at-studio-auth'));
        }

        $token_endpoint = 'https://discord.com/api/oauth2/token';

        $response = wp_remote_post($token_endpoint, [
            'body' => [
                'client_id' => $this->client_id,
                'client_secret' => $this->client_secret,
                'grant_type' => 'authorization_code',
                'code' => $code,
                'redirect_uri' => $this->redirect_uri
            ]
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (isset($data['access_token'])) {
            return $this->get_user_data($data['access_token']);
        }

        return false;
    }

    /**
     * الحصول على بيانات المستخدم
     */
    private function get_user_data($access_token) {
        $user_endpoint = 'https://discord.com/api/users/@me';
        $response = wp_remote_get($user_endpoint, [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token
            ]
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        return json_decode(wp_remote_retrieve_body($response), true);
    }

    /**
     * تسجيل دخول المستخدم أو إنشاء حساب جديد
     */
    public function login_or_register_user($user_data) {
        // البحث عن مستخدم موجود بالبريد الإلكتروني
        $user = get_user_by('email', $user_data['email']);

        if (!$user) {
            // إنشاء مستخدم جديد
            $username = sanitize_user($user_data['username']);
            $password = wp_generate_password(12, true);
            $user_id = wp_create_user($username, $password, $user_data['email']);

            if (is_wp_error($user_id)) {
                return false;
            }

            $user = get_user_by('id', $user_id);
        }

        // تحديث صورة الملف الشخصي إذا وجدت
        if (isset($user_data['avatar'])) {
            $avatar_url = "https://cdn.discordapp.com/avatars/{$user_data['id']}/{$user_data['avatar']}.png";
            update_user_meta($user->ID, 'at_studio_auth_avatar', $avatar_url);
            
            // استيراد الصورة إلى مكتبة الوسائط
            if (class_exists('AT_Studio_Avatar_Importer')) {
                AT_Studio_Avatar_Importer::import_avatar($user->ID, $avatar_url, 'discord');
            }
        }

        // تسجيل دخول المستخدم
        wp_set_auth_cookie($user->ID, true);
        return true;
    }
}
