<?php
/**
 * نظام مصادقة Google لـ WordPress
 * 
 * @package AT_Studio_Auth
 */

if (!defined('ABSPATH')) exit;

class AT_Studio_Google_Auth {
    // بيانات تطبيق Google
    private $client_id;
    private $client_secret;
    private $redirect_uri;

    public function __construct() {
        $this->client_id = get_option('at_studio_google_client_id');
        $this->client_secret = get_option('at_studio_google_client_secret');
        $this->redirect_uri = site_url('/?action=google-callback');
    }

    /**
     * الحصول على رابط مصادقة Google
     */
    public function get_auth_url() {
        $endpoint = 'https://accounts.google.com/o/oauth2/v2/auth';
        
        return $endpoint . '?' . http_build_query([
            'client_id' => $this->client_id,
            'redirect_uri' => $this->redirect_uri,
            'response_type' => 'code',
            'scope' => 'email profile',
            'access_type' => 'online',
            'prompt' => 'consent'
        ]);
    }

    /**
     * معالجة رد المصادقة
     */
    public function handle_callback($code) {
        // التحقق من وجود بيانات الاعتماد
        if (empty($this->client_id) || empty($this->client_secret)) {
            wp_die(__('بيانات تطبيق Google غير مكتملة. يرجى التحقق من الإعدادات.', 'at-studio-auth'));
        }

        $token_endpoint = 'https://oauth2.googleapis.com/token';

        $response = wp_remote_post($token_endpoint, [
            'body' => [
                'client_id' => $this->client_id,
                'client_secret' => $this->client_secret,
                'code' => $code,
                'grant_type' => 'authorization_code',
                'redirect_uri' => $this->redirect_uri
            ]
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (isset($data['access_token'])) {
            return $this->get_user_data($data['access_token']);
        }

        return false;
    }

    /**
     * الحصول على بيانات المستخدم
     */
    private function get_user_data($access_token) {
        $user_endpoint = 'https://www.googleapis.com/oauth2/v3/userinfo';
        $response = wp_remote_get($user_endpoint, [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token
            ]
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        return json_decode(wp_remote_retrieve_body($response), true);
    }

    /**
     * تسجيل دخول المستخدم أو إنشاء حساب جديد
     */
    public function login_or_register_user($user_data) {
        // استخدام البريد الإلكتروني كمعرف رئيسي
        $email = $user_data['email'];
        $user = get_user_by('email', $email);

        if (!$user) {
            // إنشاء مستخدم جديد
            $username = sanitize_user($user_data['given_name'] . '_' . $user_data['family_name']);
            $password = wp_generate_password(12, true);
            $user_id = wp_create_user($username, $password, $email);

            if (is_wp_error($user_id)) {
                return false;
            }

            // تحديث بيانات المستخدم
            wp_update_user([
                'ID' => $user_id,
                'first_name' => $user_data['given_name'],
                'last_name' => $user_data['family_name']
            ]);

            $user = get_user_by('id', $user_id);
        }

        // تحديث صورة الملف الشخصي إذا وجدت
        if (isset($user_data['picture'])) {
            update_user_meta($user->ID, 'at_studio_auth_avatar', $user_data['picture']);
            
            // استيراد الصورة إلى مكتبة الوسائط
            if (class_exists('AT_Studio_Avatar_Importer')) {
                AT_Studio_Avatar_Importer::import_avatar($user->ID, $user_data['picture'], 'google');
            }
        }

        // تسجيل دخول المستخدم
        wp_set_auth_cookie($user->ID, true);
        return true;
    }
}
